﻿"""
This is a simple implementation of a "fly to" transport.

The fly to transport works like a grappling hook, a ray is shot from the transport's
location and attaches to a location indicated by the start node node. The start node
node provides orientation and a virtual line is intersected with the scene to get the
attachment point.
"""

import math
import pickle

import viz
import vizact
import vizinfo
import vizmat
import vizshape

from transportation import Transport
from tools import ray_caster


class GrapplingHook(Transport):
	"""Creates a new instance of the fly to transport.
	"""
	def __init__(self, startNode=None, ray=None, endNode=None, **kwargs):
		super(GrapplingHook, self).__init__(**kwargs)
		
		self._percentagePerSecond = 0.5
		self._maxSpeed = 0.0
		self._speed = 0.0
		
		self._recast = False
		self._fired = False
		self._blocked = False
		self._target = None
		self._offsetShift = [0,0,0.5]
		
		# make a default start node if we don't have one already
		self._defaultStartNode = None
		if startNode is None:
			startNode = viz.addGroup()
			self._defaultStartNode = startNode
		self._startNode = startNode
		
		# make the default ray if there isn't one specified
		if ray is None:
			ray = ray_caster.SimpleRay()
			self._defaultRay = ray
		self._ray = ray
		
		# make the end point, the end point in this case just has to be a node
		if endNode is None:
			plane = vizshape.addPlane((0.25, 0.25), axis=vizshape.AXIS_Z)
			plane.color(0.0, 0.5, 0.0)
			plane.alpha(.7)
			plane.disable(viz.LIGHTING)
			plane.disable(viz.DEPTH_TEST)
			plane.drawOrder(10000)
			endNode = plane
			self._defaultEndNode = endNode
		self._endNode = endNode
		
		# add a late update for the ray so that it's matched to the view
		self._castUpdate = vizact.onupdate(100, self._updateRay)
	
	def setRay(self, ray):
		"""
		"""
		if self._defaultRay:
			self._defaultRay.remove()
			self._defaultRay = None
		self._ray = ray
	
	def setStartNode(self, startNode):
		"""
		"""
		if self._defaultStartNode:
			self._defaultStartNode.remove()
			self._defaultStartNode = None
		self._startNode = startNode
	
	def setEndNode(self, endNode):
		"""
		"""
		if self._defaultEndNode:
			self._defaultEndNode.remove()
			self._defaultEndNode = None
		self._endNode = endNode
	
	def _updateRay(self):
		if self._recast:
			self.clear()
			intersection = self._intersect()
			if intersection:
				self._assignLookAt(intersection, self._endNode)
				self._endNode.visible(True)
			self._target = intersection
			
			# get the maximum movement speed using the distance between the current and far point and the percentage per second of that distance we want to travel.
			if self._pivot is None:
				vec = vizmat.Vector(self._target.point) - self.getPosition(viz.ABS_GLOBAL)
			else:
				vec = vizmat.Vector(self._target.point) - self._pivot.getPosition(viz.ABS_GLOBAL)
			length = vec.length()
			self._maxSpeed = length * self._percentagePerSecond
		self._recast = False
	
	def _getLookAt(self, intersection):
		"""Internal function that returns the appropriate transform for a node"""
		mat = vizmat.Transform()
		target = vizmat.Vector(intersection.point)+vizmat.Vector(intersection.normal)
		if intersection.normal == [0,1,0]:
			mat.makeLookAt(intersection.point, target, [0, 0, 1])
		elif intersection.normal == [0,-1,0]:
			mat.makeLookAt(intersection.point, target, [0, 0, 1])
		else:
			mat.makeLookAt(intersection.point, target, [0, 1, 0])
		return mat
	
	def _assignLookAt(self, intersection, node):
		"""Internal function used to make a node face the appropriate direction."""
		mat = self._getLookAt(intersection)
		node.setQuat(mat.getQuat())
		node.setPosition(intersection.point, viz.ABS_GLOBAL)
	
	def clear(self):
		self._endNode.visible(False)
		self._ray.visible(False)
	
	def castLine(self):
		self._recast = True
	
	def _intersect(self):
		"""Internal function that handles the intersections"""
		self._ray.visible(False)
		
		if self._pivot is None:
			p1 = self.getPosition(viz.ABS_GLOBAL)
		else:
			p1 = self._pivot.getPosition(viz.ABS_GLOBAL)
		
		# get the orientation from the start node node
		mat = self._startNode.getMatrix(viz.ABS_GLOBAL)
		
		targetPos = vizmat.Vector(mat.preMultVec([0, 0, 100]))
		offsetStartingPos = vizmat.Vector(mat.preMultVec(self._offsetShift))
		
		intersection = viz.intersect(offsetStartingPos, targetPos, ignoreBackFace=True)
		
		if intersection:
			p2 = intersection.point
		else:
			p2 = targetPos
		
		self._ray.setStart(p1)
		self._ray.setEnd(p2)
		
		self._ray.visible(True)
		
		return intersection
	
	def reelIn(self, mag=1):
		"""Starts the user moving toward the selected destination
		"""
		self._speed = self._maxSpeed*mag
		if self._endNode.getVisible():
			self._targetLocation = vizmat.Vector(self._endNode.getPosition(viz.ABS_GLOBAL))
		if not self._deferred:
			self.finalize()
	
	def finalize(self):
		self._updateTime()
		
		if self._target is not None:
			pos = self.getPosition(viz.ABS_GLOBAL)
			if self._pivot is None:
				vec = vizmat.Vector(self._target.point) - pos
			else:
				vec = vizmat.Vector(self._target.point) - self._pivot.getPosition(viz.ABS_GLOBAL)
			
			length = vec.length()
			
			if length > 0.001:
				vec.normalize()
				vec = vec*min(length, self._speed*self._dt)
				self.setPosition(vec+pos, viz.ABS_GLOBAL)
			else:
				self._target = None
				self.clear()
		
		self._speed = 0
	
	def remove(self):
		Transport.remove(self)
		self.setRay(None)# removes default
		self.setStartNode(None)# removes default
		self.setEndNode(None)# removes default
		self._castUpdate.setEnabled(False)
		self._castUpdate.remove()


